function [varargout] = gplvm(hyp, inf, mean, cov, lik, y)

% Gaussian Process Latent Variable Model.
% Modified version of the original gp.m function of Rasmussen. It has
% been changed to account for multi-dimensional output y (shared covariance
% function among the dimensions). It also accounts for multiple output
% spaces. 
%
%   training: [nlZ dnlZ] = gplvm(hyp, inf, mean, cov, lik, y);
%
% where:
%
%   hyp      column vector of latent variables and hyperparameters.
%   inf      Only Exact inference is supported. @infExact_mod
%   cov      prior covariance function (see below)
%   mean     prior mean function. It is not supported yet. Keep it empty!!!
%   lik      likelihood function. Only Gaussian liklihood is supported.
%   y        v-dimensional cell  which contains the nxD inputs from v views
%
%   nlZ      returned value of the negative log marginal likelihood
%   dnlZ     column vector of partial derivatives of the negative log
%            likelihood
%
%   post     struct representation of the  posterior for each view in a
%            v-dimensional cell. 3rd output in training mode

if nargin~=6
  disp('Usage: [nlZ dnlZ          ] = gplvm(hyp, inf, mean, cov, lik, y);')
  return
end

if isempty(inf),  inf = @infExact_mod; else                    % set default inf
  if iscell(inf), inf = inf{1}; end                      % cell input is allowed
  if ischar(inf), inf = str2func(inf); end        % convert into function handle
end
if isempty(mean), mean = {@meanZero}; end                     % set default mean
if ischar(mean) || isa(mean, 'function_handle'), mean = {mean}; end  % make cell
if isempty(cov), error('Covariance function cannot be empty'); end  % no default
if ischar(cov)  || isa(cov,  'function_handle'), cov  = {cov};  end  % make cell
cov1 = cov{1}; if isa(cov1, 'function_handle'), cov1 = func2str(cov1); end
if strcmp(cov1,'covFITC'); inf = @infFITC; end       % only one possible inf alg
if isempty(lik),  lik = @likGauss; else                        % set default lik
  if iscell(lik), lik = lik{1}; end                      % cell input is allowed
  if ischar(lik), lik = str2func(lik); end        % convert into function handle
end

%%%% compute number of hyper-parameters
cov_param = eval(feval(cov{:}));
lik_param = eval(feval(lik));

%%%% Be sure y is cell
if ~iscell(y) && nargin==6
    y = {y};
end
N_diff_mod = numel(y);

%%%% Get the size of the latent space NxD
N = size(y{1}, 1);
D = (numel(hyp) - N_diff_mod * (cov_param+lik_param)) / N;

%%%% Split the latent space, x, from the hyperparameters, hyp.
x = reshape(hyp(1:N*D), N, D);
tmp_hyp = hyp(N*D+1:end);

clear hyp;
for i = 1:N_diff_mod
    hyp{i}.cov = tmp_hyp((i-1)*(cov_param+lik_param)+1:i*(cov_param+lik_param)-lik_param);
    hyp{i}.lik = tmp_hyp((i-1)*(cov_param+lik_param)+cov_param+1:i*(cov_param+lik_param));
end

% if ~iscell(hyp)
%     hyp = {hyp};
% end

for i = 1:N_diff_mod
    if ~isfield(hyp{i},'mean'), hyp{i}.mean = []; end        % check the hyp specification
    if eval(feval(mean{:})) ~= numel(hyp{i}.mean)
        error('Number of mean function hyperparameters disagree with mean function')
    end
    if ~isfield(hyp{i},'cov'), hyp{i}.cov = []; end
    if eval(feval(cov{:})) ~= numel(hyp{i}.cov)
        error('Number of cov function hyperparameters disagree with cov function')
    end
    if ~isfield(hyp{i},'lik'), hyp{i}.lik = []; end
    if eval(feval(lik)) ~= numel(hyp{i}.lik)
        error('Number of lik function hyperparameters disagree with lik function')
    end
end


try                                                  % call the inference method
  post = cell(size(hyp));
  nlZ = 0;
  
  if nargout==1
      for i = 1:N_diff_mod
          [post{i}, tmp_nlZ] = inf(hyp{i}, mean, cov, lik, x, y{i}, false); dnlZ = {};
          nlZ = nlZ + tmp_nlZ;
      end
  else
      d_hyp = []; dx = zeros(N*D,1);
      for i = 1:N_diff_mod
          [post{i}, tmp_nlZ, tmp_dnlZ] = inf(hyp{i}, mean, cov, lik, x, y{i}, true);
          nlZ = nlZ + tmp_nlZ;
          d_hyp = [d_hyp; tmp_dnlZ.cov(:); tmp_dnlZ.lik(:)];
          dx = dx + tmp_dnlZ.x(:);
      end
      dnlZ = [dx(:); d_hyp(:)];
  end
catch
      msgstr = lasterr;
      
      warning('Inference method failed [%s] .. attempting to continue',msgstr)
      dnlZ = zeros(N*D + N_diff_mod*(cov_param + lik_param), 1);
      varargout = {NaN, dnlZ}; return                    % continue with a warning
  
end

varargout = {nlZ, dnlZ, post};    % report -log marg lik, derivatives and post
